/* ==========================================
   CHINESE SENTENCE RACER - GAME LOGIC
   Educational typing game for Primary 3 students
   ========================================== */

// ==========================================
// GAME DATA - Race Levels with Pinyin
// ==========================================
const raceLevels = [
    {
        level: 1,
        text: "小明是一个懂事的孩子。",
        pinyin: "xiǎo míng shì yī gè dǒng shì de hái zi。",
        aiSpeed: 8000 // AI completes in 8 seconds
    },
    {
        level: 2,
        text: "今天是星期天，我和爸爸一起去公园跑步。",
        pinyin: "jīn tiān shì xīng qī tiān， wǒ hé bà ba yī qǐ qù gōng yuán pǎo bù。",
        aiSpeed: 12000 // AI completes in 12 seconds
    },
    {
        level: 3,
        text: "这里的风景真美！公园里有五颜六色的花朵，还有高大的树木。我们在树下休息，觉得非常开心。",
        pinyin: "zhè lǐ de fēng jǐng zhēn měi！ gōng yuán lǐ yǒu wǔ yán liù sè de huā duǒ， hái yǒu gāo dà de shù mù。 wǒ men zài shù xià xiū xi， jué de fēi cháng kāi xīn。",
        aiSpeed: 18000 // AI completes in 18 seconds
    }
];

// ==========================================
// GAME STATE
// ==========================================
let currentLevel = 0;
let userInput = "";
let aiPosition = 0;
let aiInterval = null;
let gameStarted = false;
let showPinyin = false;
let raceFinished = false;

// ==========================================
// DOM ELEMENTS
// ==========================================
const elements = {
    levelBadge: document.getElementById('levelBadge'),
    pinyinToggle: document.getElementById('pinyinToggle'),
    resetBtn: document.getElementById('resetBtn'),
    aiRacer: document.getElementById('aiRacer'),
    playerRacer: document.getElementById('playerRacer'),
    pinyinDisplay: document.getElementById('pinyinDisplay'),
    targetText: document.getElementById('targetText'),
    userInputBox: document.getElementById('userInput'),
    progressBar: document.getElementById('progressBar'),
    victoryModal: document.getElementById('victoryModal'),
    modalMessage: document.getElementById('modalMessage'),
    nextLevelBtn: document.getElementById('nextLevelBtn'),
    confettiContainer: document.getElementById('confettiContainer')
};

// ==========================================
// INITIALIZATION
// ==========================================
function init() {
    loadLevel(currentLevel);
    attachEventListeners();
}

// ==========================================
// LOAD LEVEL
// ==========================================
function loadLevel(levelIndex) {
    // Reset game state
    resetGame();
    
    const level = raceLevels[levelIndex];
    currentLevel = levelIndex;
    
    // Update UI
    elements.levelBadge.textContent = `Level ${level.level}`;
    elements.pinyinDisplay.textContent = level.pinyin;
    
    // Render target text with character spans
    renderTargetText(level.text);
    
    // Clear input
    elements.userInputBox.value = "";
    elements.userInputBox.disabled = false;
    elements.userInputBox.focus();
}

// ==========================================
// RENDER TARGET TEXT
// Creates individual spans for each character
// ==========================================
function renderTargetText(text) {
    elements.targetText.innerHTML = "";
    
    for (let i = 0; i < text.length; i++) {
        const span = document.createElement('span');
        span.className = 'char';
        span.textContent = text[i];
        span.dataset.index = i;
        elements.targetText.appendChild(span);
    }
    
    // Highlight first character as current
    if (elements.targetText.children.length > 0) {
        elements.targetText.children[0].classList.add('current');
    }
}

// ==========================================
// ATTACH EVENT LISTENERS
// ==========================================
function attachEventListeners() {
    // Input handling with IME support
    elements.userInputBox.addEventListener('input', handleInput);
    
    // Pinyin toggle
    elements.pinyinToggle.addEventListener('click', togglePinyin);
    
    // Reset button
    elements.resetBtn.addEventListener('click', () => loadLevel(currentLevel));
    
    // Next level button
    elements.nextLevelBtn.addEventListener('click', handleNextLevel);
}

// ==========================================
// HANDLE USER INPUT
// Real-time validation with IME support
// ==========================================
function handleInput(e) {
    // Start AI race on first input
    if (!gameStarted && !raceFinished) {
        startAIRace();
        gameStarted = true;
    }
    
    const level = raceLevels[currentLevel];
    const targetText = level.text;
    userInput = e.target.value;
    
    // Validate each character
    let allCorrect = true;
    let incorrectFound = false;
    
    for (let i = 0; i < targetText.length; i++) {
        const charSpan = elements.targetText.children[i];
        
        if (i < userInput.length) {
            // Check if character matches
            if (userInput[i] === targetText[i]) {
                charSpan.className = 'char correct';
            } else {
                charSpan.className = 'char incorrect';
                allCorrect = false;
                incorrectFound = true;
            }
        } else {
            // Not yet typed
            charSpan.className = 'char';
            if (i === userInput.length) {
                charSpan.classList.add('current');
            }
            allCorrect = false;
        }
    }
    
    // Update player position based on correct progress
    const correctLength = getCorrectLength(userInput, targetText);
    const progress = (correctLength / targetText.length) * 100;
    updatePlayerPosition(progress);
    elements.progressBar.style.width = progress + '%';
    
    // Check for completion
    if (userInput === targetText && allCorrect) {
        handleVictory();
    }
}

// ==========================================
// GET CORRECT LENGTH
// Returns how many characters are correct from start
// ==========================================
function getCorrectLength(input, target) {
    let count = 0;
    for (let i = 0; i < Math.min(input.length, target.length); i++) {
        if (input[i] === target[i]) {
            count++;
        } else {
            break; // Stop at first incorrect character
        }
    }
    return count;
}

// ==========================================
// UPDATE PLAYER POSITION
// ==========================================
function updatePlayerPosition(percentage) {
    const maxPosition = elements.playerRacer.parentElement.offsetWidth - 40;
    const position = (percentage / 100) * maxPosition;
    elements.playerRacer.style.left = position + 'px';
}

// ==========================================
// START AI RACE
// AI moves at constant speed based on level
// ==========================================
function startAIRace() {
    const level = raceLevels[currentLevel];
    const duration = level.aiSpeed;
    const fps = 60;
    const totalFrames = (duration / 1000) * fps;
    let frame = 0;
    
    aiInterval = setInterval(() => {
        frame++;
        const progress = (frame / totalFrames) * 100;
        
        if (progress >= 100) {
            updateAIPosition(100);
            clearInterval(aiInterval);
            
            // AI wins if player hasn't finished
            if (!raceFinished) {
                handleDefeat();
            }
        } else {
            updateAIPosition(progress);
        }
    }, 1000 / fps);
}

// ==========================================
// UPDATE AI POSITION
// ==========================================
function updateAIPosition(percentage) {
    const maxPosition = elements.aiRacer.parentElement.offsetWidth - 40;
    const position = (percentage / 100) * maxPosition;
    elements.aiRacer.style.left = position + 'px';
}

// ==========================================
// HANDLE VICTORY
// Player wins the race
// ==========================================
function handleVictory() {
    raceFinished = true;
    clearInterval(aiInterval);
    elements.userInputBox.disabled = true;
    
    // Show victory modal
    elements.modalMessage.textContent = "你赢了！You won the race!";
    elements.victoryModal.classList.add('show');
    
    // Create confetti
    createConfetti();
}

// ==========================================
// HANDLE DEFEAT
// AI wins the race
// ==========================================
function handleDefeat() {
    raceFinished = true;
    elements.userInputBox.disabled = true;
    
    // Show defeat modal
    elements.modalMessage.textContent = "乌龟赢了！Try again!";
    elements.nextLevelBtn.textContent = "Try Again 🔄";
    elements.victoryModal.classList.add('show');
}

// ==========================================
// HANDLE NEXT LEVEL
// ==========================================
function handleNextLevel() {
    elements.victoryModal.classList.remove('show');
    elements.confettiContainer.innerHTML = '';
    elements.nextLevelBtn.textContent = "Next Level ➡️";
    
    // If player won, advance to next level
    if (elements.modalMessage.textContent.includes("You won")) {
        if (currentLevel < raceLevels.length - 1) {
            loadLevel(currentLevel + 1);
        } else {
            // Game completed
            currentLevel = 0;
            loadLevel(0);
        }
    } else {
        // Retry current level
        loadLevel(currentLevel);
    }
}

// ==========================================
// TOGGLE PINYIN DISPLAY
// ==========================================
function togglePinyin() {
    showPinyin = !showPinyin;
    
    if (showPinyin) {
        elements.pinyinDisplay.classList.add('show');
        elements.pinyinToggle.classList.add('active');
    } else {
        elements.pinyinDisplay.classList.remove('show');
        elements.pinyinToggle.classList.remove('active');
    }
}

// ==========================================
// RESET GAME
// ==========================================
function resetGame() {
    clearInterval(aiInterval);
    gameStarted = false;
    raceFinished = false;
    userInput = "";
    aiPosition = 0;
    
    // Reset positions
    elements.aiRacer.style.left = '0px';
    elements.playerRacer.style.left = '0px';
    elements.progressBar.style.width = '0%';
}

// ==========================================
// CREATE CONFETTI EFFECT
// Simple CSS-based confetti animation
// ==========================================
function createConfetti() {
    const colors = ['#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#ffd89b'];
    const confettiCount = 50;
    
    for (let i = 0; i < confettiCount; i++) {
        const confetti = document.createElement('div');
        confetti.className = 'confetti';
        confetti.style.left = Math.random() * 100 + '%';
        confetti.style.background = colors[Math.floor(Math.random() * colors.length)];
        confetti.style.animationDelay = Math.random() * 3 + 's';
        confetti.style.animationDuration = (Math.random() * 2 + 2) + 's';
        elements.confettiContainer.appendChild(confetti);
    }
}

// ==========================================
// START THE GAME
// ==========================================
init();