'use strict';

// xAPI Integrator - Pure client-side (no server)
// - Reads an uploaded ZIP with JSZip
// - Finds the folder containing index.html
// - Injects xAPI libs and optional instrumentation snippet
// - Removes CSP-blocked GA/AdSense scripts
// - Adds vendored libs into lib/ within the content folder
// - Re-zips and offers a download link

(function () {
  const zipInput = document.getElementById('zipInput');
  const integrateBtn = document.getElementById('integrateBtn');
  const downloadLink = document.getElementById('downloadLink');
  const statusText = document.getElementById('statusText');
  const logEl = document.getElementById('log');
  const dropZone = document.getElementById('dropZone');
  const fileName = document.getElementById('fileName');

  function log(msg) {
    try { console.log('[integrator]', msg); } catch {}
    if (logEl) {
      logEl.textContent += (logEl.textContent ? '\n' : '') + msg;
    }
  }

  function setStatus(msg) {
    if (statusText) statusText.textContent = msg || '';
  }

  // Drag and drop functionality
  async function handleFile(file) {
    if (file && (file.type === 'application/zip' || file.name.endsWith('.zip'))) {
      // Create a new FileList-like object and assign to input
      const dataTransfer = new DataTransfer();
      dataTransfer.items.add(file);
      zipInput.files = dataTransfer.files;
      
      if (fileName) {
        fileName.textContent = `📦 ${file.name}`;
      }
      log(`File selected: ${file.name}`);
      
      // Immediately analyze and show recommendation
      await analyzeAndRecommendMode(file);
    } else {
      setStatus('Please drop a ZIP file.');
    }
  }

  if (dropZone) {
    // Prevent default drag behaviors
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
      dropZone.addEventListener(eventName, (e) => {
        e.preventDefault();
        e.stopPropagation();
      }, false);
    });

    // Highlight drop zone when dragging over it
    ['dragenter', 'dragover'].forEach(eventName => {
      dropZone.addEventListener(eventName, () => {
        dropZone.classList.add('drag-over');
      }, false);
    });

    ['dragleave', 'drop'].forEach(eventName => {
      dropZone.addEventListener(eventName, () => {
        dropZone.classList.remove('drag-over');
      }, false);
    });

    // Handle dropped files
    dropZone.addEventListener('drop', async (e) => {
      const files = e.dataTransfer.files;
      if (files.length > 0) {
        try {
          await handleFile(files[0]);
        } catch (err) {
          console.error('Drop error:', err);
          setStatus('Error processing dropped file');
        }
      }
    }, false);

    // Handle click to browse
    dropZone.addEventListener('click', () => {
      zipInput.click();
    });
  }

  // Handle file input change with immediate analysis
  if (zipInput) {
    zipInput.addEventListener('change', async (e) => {
      const file = e.target.files[0];
      if (file && fileName) {
        fileName.textContent = `📦 ${file.name}`;
      }
      
      // Immediately analyze and show recommendation
      if (file && file.name.endsWith('.zip')) {
        await analyzeAndRecommendMode(file);
      }
    });
  }

  // Analyze ZIP and show recommendation popup
  async function analyzeAndRecommendMode(file) {
    try {
      setStatus('Analyzing uploaded file...');
      
      if (!window.JSZip) {
        setStatus('');
        return;
      }
      
      const jszip = new JSZip();
      const zip = await jszip.loadAsync(file);
      
      const { contentDir, indexPath } = findContentDirFromZip(zip);
      if (!indexPath) {
        setStatus('');
        return;
      }
      
      const html = await zip.file(indexPath).async('string');
      
      // Try to load JavaScript file for analysis
      let jsContent = '';
      const jsFiles = Object.keys(zip.files).filter(k => k.endsWith('.js') && !k.includes('node_modules'));
      if (jsFiles.length > 0) {
        try {
          jsContent = await zip.file(jsFiles[0]).async('string');
        } catch (e) {
          // Silent fail
        }
      }
      
      // Analyze content
      const analysis = await analyzeInteractive(html, jsContent);
      
      setStatus('');
      
      // Show recommendation popup
      const modeNames = {
        'timeline': 'Timeline',
        'quiz': 'Quiz',
        'minimal': 'Minimal'
      };
      
      const currentMode = getSelectedMode();
      const recommendedMode = analysis.recommendedMode;
      
      let message = `📊 SMART MODE DETECTION\n\n`;
      message += `File analyzed: ${file.name}\n\n`;
      message += `✓ Recommended Mode: ${modeNames[recommendedMode].toUpperCase()}\n`;
      message += `✓ Confidence: ${analysis.confidence}%\n\n`;
      message += `Detected Features:\n`;
      if (analysis.hasRadioButtons) message += `  • Radio buttons (quiz questions)\n`;
      if (analysis.hasCheckboxes) message += `  • Checkboxes (multiple choice)\n`;
      if (analysis.hasTextInputs) message += `  • Text inputs (short answer)\n`;
      if (analysis.hasDragDrop) message += `  • Drag & drop interactions\n`;
      if (analysis.hasGameState) message += `  • Game state tracking\n`;
      if (analysis.hasScore) message += `  • Scoring system\n`;
      if (analysis.hasCanvas) message += `  • Canvas elements\n`;
      if (analysis.hasPlayPause) message += `  • Play/pause controls\n`;
      
      if (analysis.reasons.length === 0) {
        message += `  • No specific patterns detected\n`;
      }
      
      message += `\n`;
      
      if (currentMode !== recommendedMode && analysis.confidence >= 50) {
        message += `Currently selected: ${modeNames[currentMode]}\n\n`;
        message += `Would you like to switch to ${modeNames[recommendedMode]} mode?\n`;
        message += `(This mode has the best chance of success for this content)`;
        
        const shouldSwitch = confirm(message);
        
        if (shouldSwitch) {
          // Update the radio button
          document.querySelectorAll('input[name="mode"]').forEach(radio => {
            if (radio.value === recommendedMode) {
              radio.checked = true;
            }
          });
          log(`Mode automatically switched to: ${recommendedMode.toUpperCase()} (recommended)`);
        } else {
          log(`User kept selected mode: ${currentMode.toUpperCase()}`);
        }
      } else {
        message += `Current selection: ${modeNames[currentMode]}\n`;
        if (currentMode === recommendedMode) {
          message += `\n✓ Your selection matches the recommendation!`;
        } else {
          message += `\nNote: ${modeNames[recommendedMode]} mode is recommended, but you can proceed with ${modeNames[currentMode]} if preferred.`;
        }
        alert(message);
        log(`Analysis complete. Recommended: ${recommendedMode.toUpperCase()}, Selected: ${currentMode.toUpperCase()}`);
      }
      
    } catch (e) {
      console.error('Analysis error:', e);
      setStatus('');
    }
  }

  function getSelectedMode() {
    const el = document.querySelector('input[name="mode"]:checked');
    return (el && el.value) || 'timeline';
  }

  function shouldKeepAnalytics() {
    const checkbox = document.getElementById('keepAnalytics');
    return checkbox ? checkbox.checked : false;
  }

  function joinPath(...parts) {
    return parts
      .filter(Boolean)
      .map(p => String(p).replace(/^\/+|\/+$/g, ''))
      .filter(p => p.length > 0)
      .join('/')
      + (String(parts[parts.length - 1]).endsWith('/') ? '/' : '');
  }

  // Load vendor libs from ./vendor so we can place them into the output ZIP
  async function loadVendors() {
    async function fetchText(url) {
      const res = await fetch(url);
      if (!res.ok) throw new Error(`Failed to fetch ${url}: ${res.status} ${res.statusText}`);
      return await res.text();
    }

    // In file:// contexts, fetch may be blocked. Advise to serve over http(s).
    try {
      const [wrapper, glue] = await Promise.all([
        fetchText('./vendor/xapiwrapper.min.js'),
        fetchText('./vendor/xAPI.js'),
      ]);
      return { wrapper, glue };
    } catch (e) {
      // Fallback: try vendor files from ../assets/lib relative to public/
      try {
        const [wrapper2, glue2] = await Promise.all([
          fetchText('../assets/lib/xapiwrapper.min.js'),
          fetchText('../assets/lib/xAPI.js'),
        ]);
        log('Loaded vendor scripts from ../assets/lib fallback.');
        return { wrapper: wrapper2, glue: glue2 };
      } catch (e2) {
        setStatus('Failed to load vendor scripts. If you opened this page via file://, please serve over HTTP(S) (e.g., npx http-server xapi-integrator/public) or deploy to any static host.');
        log('Error loading vendor scripts. If opening via file://, run a local web server (e.g., `npx http-server`).');
        throw e;
      }
    }
  }

  // Find content dir containing index.html (root, single subfolder, or shallow search)
  function findContentDirFromZip(zip) {
    const files = Object.keys(zip.files); // paths with forward slashes
    const indexCandidates = files.filter(k => k.toLowerCase().endsWith('index.html'));

    if (indexCandidates.length === 0) return { contentDir: '', indexPath: null };

    // Choose the shortest (shallowest) path
    const sorted = indexCandidates.sort((a, b) => {
      const da = a.split('/').length;
      const db = b.split('/').length;
      if (da !== db) return da - db;
      return a.length - b.length;
    });

    const indexPath = sorted[0];
    const dir = indexPath.slice(0, indexPath.length - 'index.html'.length);
    return { contentDir: dir, indexPath };
  }

  function serializeDocument(doc) {
    // Preserve a doctype if one exists; if not, prefix with <!doctype html>
    const doctype = '<!doctype html>';
    return doctype + '\n' + doc.documentElement.outerHTML;
  }

  function ensureHeadBody(doc) {
    if (!doc.head) {
      const head = doc.createElement('head');
      const html = doc.documentElement || doc.querySelector('html');
      if (html) html.insertBefore(head, html.firstChild);
    }
    if (!doc.body) {
      const body = doc.createElement('body');
      const html = doc.documentElement || doc.querySelector('html');
      if (html) html.appendChild(body);
    }
  }

  function hasScriptBySrc(doc, endsWithPath) {
    const scripts = Array.from(doc.querySelectorAll('script[src]'));
    return scripts.some(s => {
      const src = s.getAttribute('src') || '';
      return src.endsWith(endsWithPath) || src.includes(endsWithPath);
    });
  }

  function injectScriptsIntoHtml(html, options) {
    const { mode = 'timeline', keepAnalytics = false } = options || {};
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');
    ensureHeadBody(doc);

    // Remove CSP-blocked scripts (unless user opts to keep them)
    if (!keepAnalytics) {
      Array.from(doc.querySelectorAll('script[src*="googletagmanager.com"],script[src*="googlesyndication.com"]'))
        .forEach(s => s.remove());
    }

    // Inject xAPI libs if missing
    if (!hasScriptBySrc(doc, 'lib/xapiwrapper.min.js')) {
      const s1 = doc.createElement('script');
      s1.setAttribute('src', './lib/xapiwrapper.min.js');
      doc.head.appendChild(s1);
    }
    if (!hasScriptBySrc(doc, 'lib/xAPI.js')) {
      const s2 = doc.createElement('script');
      s2.setAttribute('src', './lib/xAPI.js');
      doc.head.appendChild(s2);
    }

    // Inject instrumentation snippet (timeline or quiz mode)
    if (mode === 'timeline' || mode === 'quiz') {
      const already = Array.from(doc.querySelectorAll('script'))
        .some(s => ((s.textContent || '').includes('__xapiIntegrator')));
      if (!already) {
        const snippet = mode === 'quiz' ? `
(function(){
  if (window.__xapiIntegrator) return;
  window.__xapiIntegrator = true;

  const sessionStart = Date.now();
  const quizData = {
    questions: [],
    currentQuestion: null,
    totalScore: 0,
    maxScore: 0,
    attempts: [],
    navigationLog: []
  };

  // Pending payload
  window.__xapiPending = null;

  function tSeconds() { return Math.round((Date.now() - sessionStart)/1000); }

  function logQuizAction(action, details) {
    const t = tSeconds();
    const entry = { t, action, details };
    quizData.navigationLog.push(entry);
    try { console.log('[xAPI Quiz]', 't=' + t + ',', action, details); } catch(e){}
    updateQuizPanel();
    setPending('quiz-action');
  }

  // Detect and track questions
  function detectQuestions() {
    // Look for common quiz patterns: radio groups, checkbox groups, numbered divs
    const radioGroups = {};
    document.querySelectorAll('input[type="radio"]').forEach(radio => {
      const name = radio.name;
      if (name && !radioGroups[name]) {
        radioGroups[name] = {
          type: 'single-choice',
          name: name,
          options: [],
          selected: null,
          startTime: Date.now(),
          attempts: 0
        };
        quizData.questions.push(radioGroups[name]);
      }
      if (name) radioGroups[name].options.push(radio.value || radio.id || 'option');
    });

    // Look for checkbox groups
    const checkboxGroups = {};
    document.querySelectorAll('input[type="checkbox"]').forEach(cb => {
      const name = cb.name || 'checkboxes';
      if (!checkboxGroups[name]) {
        checkboxGroups[name] = {
          type: 'multiple-choice',
          name: name,
          options: [],
          selected: [],
          startTime: Date.now(),
          attempts: 0
        };
        quizData.questions.push(checkboxGroups[name]);
      }
      checkboxGroups[name].options.push(cb.value || cb.id || 'option');
    });

    // Look for text inputs (short answer)
    document.querySelectorAll('input[type="text"], textarea').forEach(input => {
      const id = input.id || input.name || 'text-' + Math.random().toString(36).substr(2, 9);
      if (!input.dataset.quizTracked) {
        input.dataset.quizTracked = 'true';
        quizData.questions.push({
          type: 'text-input',
          name: id,
          value: '',
          startTime: Date.now(),
          attempts: 0
        });
      }
    });

    quizData.maxScore = quizData.questions.length;
    logQuizAction('questions-detected', { count: quizData.questions.length });
  }

  // Build comprehensive quiz feedback
  function buildQuizFeedback() {
    const elapsedSeconds = tSeconds();
    const attempted = quizData.questions.filter(q => q.selected || (q.selected && q.selected.length > 0) || q.value).length;
    
    let questionsHtml = quizData.questions.map((q, idx) => {
      const num = idx + 1;
      const timeSpent = q.startTime ? Math.round((Date.now() - q.startTime) / 1000) : 0;
      let answer = 'Not answered';
      if (q.type === 'single-choice' && q.selected) answer = q.selected;
      if (q.type === 'multiple-choice' && q.selected && q.selected.length > 0) answer = q.selected.join(', ');
      if (q.type === 'text-input' && q.value) answer = q.value;
      return 'Q' + num + ' (' + q.type + '): ' + answer + ' [' + timeSpent + 's, ' + q.attempts + ' attempts]';
    }).join('<br>');

    const navLog = quizData.navigationLog.slice(-10).map(n => 
      't=' + n.t + ': ' + n.action + (n.details ? ' - ' + JSON.stringify(n.details) : '')
    ).join('<br>');

    return '<div><div>'
      + '<strong>Quiz Progress</strong><br>'
      + 'Questions: ' + attempted + '/' + quizData.questions.length + '<br>'
      + 'Score: ' + quizData.totalScore + '/' + quizData.maxScore + '<br>'
      + 'Time: ' + elapsedSeconds + 's<br><br>'
      + '<strong>Questions Detail:</strong><br>' + questionsHtml + '<br><br>'
      + '<strong>Recent Actions:</strong><br>' + navLog
      + '</div></div>';
  }

  // Floating quiz analytics panel
  const panel = document.createElement('div');
  Object.assign(panel.style, {
    position: 'fixed', right: '12px', bottom: '56px',
    minWidth: '280px', maxWidth: '380px',
    background: 'rgba(17,24,39,0.95)', color: '#e5e7eb',
    border: '1px solid #374151', borderRadius: '8px',
    padding: '10px 12px', zIndex: 999998, fontSize: '12px', lineHeight: '1.4',
    maxHeight: '60vh', overflowY: 'auto'
  });
  panel.setAttribute('id', 'xapiQuizPanel');

  function updateQuizPanel() {
    try {
      const elapsedSeconds = tSeconds();
      const attempted = quizData.questions.filter(q => 
        q.selected || (q.selected && q.selected.length > 0) || q.value
      ).length;
      
      panel.innerHTML = ''
        + '<div style="font-weight:600;margin-bottom:8px;color:#60a5fa;">📊 Quiz Analytics</div>'
        + '<div>Questions: ' + attempted + '/' + quizData.questions.length + '</div>'
        + '<div>Score: ' + quizData.totalScore + '/' + quizData.maxScore + '</div>'
        + '<div>Time: ' + elapsedSeconds + 's</div>'
        + '<div style="margin-top:8px;padding-top:8px;border-top:1px solid #374151;font-size:11px;">'
        + 'Actions: ' + quizData.navigationLog.length + '</div>';
    } catch(e){}
  }

  function setPending(reason) {
    try {
      const feedback = buildQuizFeedback();
      window.__xapiPending = {
        score: quizData.totalScore,
        maxScore: quizData.maxScore,
        feedback,
        reason,
        quizData: JSON.parse(JSON.stringify(quizData))
      };
    } catch(e){}
  }

  // Track radio button selections
  document.addEventListener('change', (e) => {
    const el = e.target;
    if (el && el.type === 'radio' && el.name) {
      const q = quizData.questions.find(q => q.name === el.name);
      if (q) {
        q.selected = el.value || el.id || 'selected';
        q.attempts++;
        logQuizAction('answer-selected', { question: el.name, answer: q.selected, type: 'radio' });
      }
    } else if (el && el.type === 'checkbox') {
      const name = el.name || 'checkboxes';
      const q = quizData.questions.find(q => q.name === name);
      if (q) {
        if (!q.selected) q.selected = [];
        const val = el.value || el.id || 'checked';
        if (el.checked) {
          if (!q.selected.includes(val)) q.selected.push(val);
        } else {
          q.selected = q.selected.filter(v => v !== val);
        }
        q.attempts++;
        logQuizAction('answer-toggled', { question: name, answer: val, checked: el.checked, type: 'checkbox' });
      }
    } else if (el && (el.type === 'text' || el.tagName === 'TEXTAREA')) {
      const id = el.id || el.name;
      const q = quizData.questions.find(q => q.name === id);
      if (q) {
        q.value = el.value;
        q.attempts++;
        logQuizAction('text-entered', { question: id, length: el.value.length, type: 'text' });
      }
    } else if (el && el.tagName === 'SELECT') {
      logQuizAction('dropdown-changed', { value: el.value, id: el.id || el.name });
    }
  }, true);

  // Track all clicks for navigation and button presses
  document.addEventListener('click', (e) => {
    const el = e.target;
    if (el && el.tagName === 'BUTTON') {
      const text = el.innerText.trim().toLowerCase();
      if (text.includes('submit') || text.includes('finish') || text.includes('done')) {
        logQuizAction('submit-clicked', { button: el.innerText.trim() });
        sendQuizState('submit');
      } else if (text.includes('next')) {
        logQuizAction('next-clicked', {});
      } else if (text.includes('prev') || text.includes('back')) {
        logQuizAction('previous-clicked', {});
      } else if (text.includes('check') || text.includes('verify')) {
        logQuizAction('check-answer', { button: el.innerText.trim() });
      } else {
        logQuizAction('button-clicked', { button: el.innerText.trim() });
      }
    }
  }, true);

  // Dedup-safe sender
  let lastSignature = null;
  function signatureOf(p) {
    try {
      return JSON.stringify({
        s: p.score,
        q: p.quizData ? p.quizData.questions.length : 0,
        a: p.quizData ? p.quizData.navigationLog.length : 0
      });
    } catch { return String(Date.now()); }
  }

  function sendQuizState(reason) {
    try {
      const feedback = buildQuizFeedback();
      const payload = {
        score: quizData.totalScore,
        maxScore: quizData.maxScore,
        feedback,
        reason,
        quizData: JSON.parse(JSON.stringify(quizData))
      };
      const sig = signatureOf(payload);
      if (sig && sig === lastSignature) return;
      lastSignature = sig;

      if (typeof window.storeState === 'function') {
        window.storeState(payload);
        logQuizAction('auto-save', { reason });
      } else {
        console.warn('storeState not available');
      }
      setPending(reason);
    } catch(e) {
      console.warn('xAPI sendState failed:', e);
    }
  }

  // Floating "Save to SLS" button
  const btn = document.createElement('button');
  btn.textContent = 'Save Quiz to SLS';
  Object.assign(btn.style, {
    position: 'fixed', right: '12px', bottom: '12px',
    padding: '10px 16px', background: '#2563eb', color: '#fff',
    border: 'none', borderRadius: '6px', zIndex: 999999,
    cursor: 'pointer', fontWeight: '600'
  });
  btn.addEventListener('click', () => {
    if (typeof window.storeState === 'function') {
      try {
        const feedback = buildQuizFeedback();
        const payload = {
          score: quizData.totalScore,
          maxScore: quizData.maxScore,
          feedback,
          reason: 'manual-save',
          quizData: JSON.parse(JSON.stringify(quizData))
        };
        window.storeState(payload);
        lastSignature = signatureOf(payload);
        logQuizAction('manual-save', {});
        alert('Quiz progress saved successfully!');
      } catch(e) {
        console.warn('xAPI sendState failed:', e);
        alert('Failed to save quiz progress. Please try again.');
      }
    } else {
      console.warn('storeState not available');
      alert('xAPI not configured. Cannot save progress.');
    }
    setPending('manual-save');
  });

  document.addEventListener('DOMContentLoaded', () => {
    try {
      document.body.appendChild(panel);
      document.body.appendChild(btn);
      // Detect questions after a short delay to ensure DOM is ready
      setTimeout(detectQuestions, 500);
      setTimeout(updateQuizPanel, 600);
    } catch(e){}
  });

  // Auto-send hooks
  document.addEventListener('visibilitychange', () => {
    if (document.visibilityState === 'hidden') sendQuizState('visibility-hidden');
  }, true);

  window.addEventListener('pagehide', () => {
    sendQuizState('pagehide');
  }, { capture: true });

  window.addEventListener('message', (event) => {
    try {
      const data = event.data;
      const key = (data && (data.type || data.event || data.action || data.message)) || '';
      if (typeof key === 'string' && /submit/i.test(key)) {
        sendQuizState('sls-submit');
      }
    } catch(e){}
  }, false);

  setPending('init');
})();
        `.trim() : `
(function(){
  if (window.__xapiIntegrator) return;
  window.__xapiIntegrator = true;

  const sessionStart = Date.now();
  const actionLog = [];
  const xapi = { score: 0, interactionCount: 0, playStart: null, totalPlayMs: 0 };

  // Pending payload that always reflects the latest feedback/score (for auto-send hooks)
  window.__xapiPending = null;

  function tSeconds() { return Math.round((Date.now() - sessionStart)/1000); }
  function flushPlayTime() {
    if (xapi.playStart) { xapi.totalPlayMs += (Date.now() - xapi.playStart); xapi.playStart = null; }
  }
  function logAction(message) {
    const t = tSeconds();
    actionLog.push({ t, message });
    try { console.log('[xAPI Integrator]', 't=' + t + ',', message); } catch(e){}
    updateFeedbackPanel();
    setPending('live');
  }

  // Live feedback builder
  function buildFeedback() {
    const elapsedSeconds = tSeconds();
    const actionsHtml = actionLog.map(a => 't = ' + a.t + ', ' + a.message).join('<br>');
    return '<div><div>'
      + 'Interactions: ' + xapi.interactionCount + '<br>'
      + 'Total Play Time: ' + elapsedSeconds + 's<br>'
      + 'Score: ' + (xapi.score || 0) + '<br><br>'
      + 'Action Log:<br>' + actionsHtml
      + '</div></div>';
  }

  // Floating live feedback panel
  const panel = document.createElement('div');
  Object.assign(panel.style, {
    position: 'fixed', right: '12px', bottom: '56px',
    minWidth: '240px', maxWidth: '320px',
    background: 'rgba(17,24,39,0.9)', color: '#e5e7eb',
    border: '1px solid #374151', borderRadius: '8px',
    padding: '8px 10px', zIndex: 999998, fontSize: '12px', lineHeight: '1.3'
  });
  panel.setAttribute('id', 'xapiLiveFeedback');
  document.addEventListener('DOMContentLoaded', () => {
    // Append panel near the end so the Save button remains on top
    try { document.body.appendChild(panel); } catch(e){}
    updateFeedbackPanel();
  });

  function updateFeedbackPanel() {
    try {
      const elapsedSeconds = tSeconds();
      panel.innerHTML = ''
        + '<div style="font-weight:600;margin-bottom:6px;">Live Feedback</div>'
        + '<div>Interactions: ' + xapi.interactionCount + '</div>'
        + '<div>Total Play Time: ' + elapsedSeconds + 's</div>'
        + '<div>Score: ' + (xapi.score || 0) + '</div>';
    } catch(e){}
  }

  // Keep play time ticking in panel while playing
  setInterval(() => {
    if (xapi.playStart) updateFeedbackPanel();
  }, 1000);

  // Keep a pending payload up to date for auto-send triggers
  function setPending(reason) {
    try {
      const feedback = buildFeedback();
      window.__xapiPending = { score: xapi.score, feedback, reason, actions: actionLog };
    } catch(e){}
  }

  document.addEventListener('click', (e) => {
    const el = e.target;
    let label = '';
    if (el && el.tagName === 'BUTTON') {
      label = el.innerText.trim() || el.id || 'button';
      logAction('click "' + label + '"');
      xapi.interactionCount++;
    } else if (el && el.tagName === 'INPUT') {
      label = (el.value || el.name || el.id || 'input').toString();
      logAction('click input "' + label + '"');
      xapi.interactionCount++;
    }
  }, true);

  document.addEventListener('change', (e) => {
    const el = e.target;
    if (el && el.tagName === 'SELECT') {
      const val = el.value;
      const label = val ? (val.charAt(0).toUpperCase() + val.slice(1)) : 'Select';
      logAction('click "' + label + '"');
      xapi.interactionCount++;
    }
  }, true);

  let dragStart = null;
  document.addEventListener('mousedown', (e) => {
    if (e.target && e.target.tagName === 'CANVAS') {
      dragStart = { x: e.offsetX, y: e.offsetY, canvas: e.target };
    }
  }, true);
  document.addEventListener('mouseup', (e) => {
    if (dragStart && dragStart.canvas === e.target) {
      const name = 'canvas';
      logAction('drag ' + name + ' to position ' + Math.round(e.offsetX) + ',' + Math.round(e.offsetY) + '.');
      xapi.interactionCount++;
    }
    dragStart = null;
  }, true);

  function hookById(id, message, after) {
    const el = document.getElementById(id);
    if (!el) return;
    el.addEventListener('click', () => {
      logAction(message);
      if (typeof after === 'function') after();
    }, true);
  }

  hookById('playBtn', 'clicked "Play"', () => {
    if (!xapi.playStart) xapi.playStart = Date.now();
    setPending('play');
  });

  hookById('pauseBtn', 'click "Pause"', () => {
    flushPlayTime();
    setPending('pause'); // keep pending updated
    // Auto-save on pause remains as a convenience
    if (typeof window.storeState === 'function') {
      try {
        const feedback = buildFeedback();
        const payload = { score: xapi.score, feedback, reason: 'pause', actions: actionLog };
        window.storeState(payload);
        logAction('auto-save (pause)');
      } catch(e) { console.warn('xAPI sendState failed:', e); }
    }
  });

  hookById('resetBtn', 'click "Reset"', () => {
    xapi.score = (typeof xapi.score === 'number') ? xapi.score + 1 : 1;
    logAction('awarded 1 mark. total score = ' + xapi.score + '.');
    setPending('reset');
  });

  // Dedup-safe sender used by auto hooks (including SLS submit via postMessage)
  let lastSignature = null;
  function signatureOf(p) {
    try { return JSON.stringify({ s: p.score, a: (p.actions ? p.actions.length : 0), f: (p.feedback || '').length }); }
    catch { return String(Date.now()); }
  }
  function sendState(reason) {
    try {
      flushPlayTime();
      const feedback = buildFeedback();
      const payload = { score: xapi.score, feedback, reason, actions: actionLog };
      const sig = signatureOf(payload);
      if (sig && sig === lastSignature) return; // avoid duplicate identical sends
      lastSignature = sig;

      if (typeof window.storeState === 'function') {
        window.storeState(payload);
        logAction('auto-save (' + reason + ')');
      } else {
        console.warn('storeState not available; ensure xAPI scripts are loaded and platform URL params are present.');
      }
      // keep pending in sync
      setPending(reason);
    } catch(e) {
      console.warn('xAPI sendState failed:', e);
    }
  }

  // Floating "Save to SLS" button (kept as required user action)
  const btn = document.createElement('button');
  btn.textContent = 'Save to SLS';
  Object.assign(btn.style, {
    position: 'fixed', right: '12px', bottom: '12px',
    padding: '8px 12px', background: '#1976d2', color: '#fff',
    border: 'none', borderRadius: '6px', zIndex: 999999, cursor: 'pointer'
  });
  btn.addEventListener('click', () => {
    flushPlayTime();
    if (typeof window.storeState === 'function') {
      try {
        const feedback = buildFeedback();
        const payload = { score: xapi.score, feedback, reason: 'manual-save', actions: actionLog };
        window.storeState(payload);
        lastSignature = signatureOf(payload); // dedup against autos
        logAction('manual save triggered');
      } catch(e) { console.warn('xAPI sendState failed:', e); }
    } else {
      console.warn('storeState not available; ensure xAPI scripts are loaded and platform URL params are present.');
    }
    setPending('manual-save');
  });
  document.addEventListener('DOMContentLoaded', () => {
    try { document.body.appendChild(btn); } catch(e){}
  });

  // Auto-send hooks:
  // 1) When content is being hidden or navigated away
  document.addEventListener('visibilitychange', () => {
    if (document.visibilityState === 'hidden') sendState('visibility-hidden');
  }, true);
  window.addEventListener('pagehide', () => {
    sendState('pagehide');
  }, { capture: true });

  // 2) Listen for parent window messages indicating "submit" (SLS submit button)
  //    If SLS can postMessage({ type: 'submit' }) on submit, this will auto-send.
  window.addEventListener('message', (event) => {
    try {
      const data = event.data;
      const key = (data && (data.type || data.event || data.action || data.message)) || '';
      if (typeof key === 'string' && /submit/i.test(key)) {
        sendState('sls-submit');
      }
    } catch(e){}
  }, false);

  // Initialize pending once
  setPending('init');
})();
        `.trim();

        const s3 = doc.createElement('script');
        s3.textContent = snippet;
        doc.body.appendChild(s3);
      }
    }

    return serializeDocument(doc);
  }

  // Analyze ZIP content and recommend mode
  async function analyzeInteractive(html, jsContent) {
    const analysis = {
      hasRadioButtons: false,
      hasCheckboxes: false,
      hasTextInputs: false,
      hasDragDrop: false,
      hasGameState: false,
      hasScore: false,
      hasCanvas: false,
      hasPlayPause: false,
      recommendedMode: 'minimal',
      confidence: 0,
      reasons: []
    };

    // Analyze HTML
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');

    // Check for form elements
    analysis.hasRadioButtons = doc.querySelectorAll('input[type="radio"]').length > 0;
    analysis.hasCheckboxes = doc.querySelectorAll('input[type="checkbox"]').length > 0;
    analysis.hasTextInputs = doc.querySelectorAll('input[type="text"], textarea').length > 0;
    analysis.hasCanvas = doc.querySelectorAll('canvas').length > 0;

    // Check JavaScript content
    if (jsContent) {
      analysis.hasDragDrop = /drag|drop|draggable/i.test(jsContent);
      analysis.hasGameState = /gameState|game_state|state\s*=/i.test(jsContent);
      analysis.hasScore = /score|points|grade/i.test(jsContent);
      analysis.hasPlayPause = /playBtn|pauseBtn|play\(\)|pause\(\)/i.test(jsContent);
    }

    // Determine recommendation
    const quizScore = (analysis.hasRadioButtons ? 3 : 0) + 
                      (analysis.hasCheckboxes ? 3 : 0) + 
                      (analysis.hasTextInputs ? 2 : 0);
    
    const gameScore = (analysis.hasDragDrop ? 3 : 0) + 
                      (analysis.hasGameState ? 2 : 0) + 
                      (analysis.hasScore ? 2 : 0);
    
    const simulationScore = (analysis.hasCanvas ? 3 : 0) + 
                           (analysis.hasPlayPause ? 3 : 0);

    if (quizScore >= 3) {
      analysis.recommendedMode = 'quiz';
      analysis.confidence = Math.min(100, quizScore * 20);
      if (analysis.hasRadioButtons) analysis.reasons.push('Contains radio buttons (single-choice questions)');
      if (analysis.hasCheckboxes) analysis.reasons.push('Contains checkboxes (multiple-choice questions)');
      if (analysis.hasTextInputs) analysis.reasons.push('Contains text inputs (short answer questions)');
    } else if (gameScore >= 3 || simulationScore >= 3) {
      analysis.recommendedMode = 'timeline';
      analysis.confidence = Math.min(100, Math.max(gameScore, simulationScore) * 15);
      if (analysis.hasDragDrop) analysis.reasons.push('Uses drag-and-drop interactions');
      if (analysis.hasGameState) analysis.reasons.push('Has game state tracking');
      if (analysis.hasScore) analysis.reasons.push('Includes scoring system');
      if (analysis.hasCanvas) analysis.reasons.push('Uses canvas for visual elements');
      if (analysis.hasPlayPause) analysis.reasons.push('Has play/pause controls');
    } else {
      analysis.recommendedMode = 'minimal';
      analysis.confidence = 50;
      analysis.reasons.push('No specific interactive patterns detected - minimal mode recommended');
    }

    return analysis;
  }

  integrateBtn.addEventListener('click', async () => {
    setStatus('');
    downloadLink.style.display = 'none';
    downloadLink.removeAttribute('href');
    downloadLink.removeAttribute('download');
    if (logEl) { logEl.textContent = ''; }

    const file = (zipInput && zipInput.files && zipInput.files[0]) ? zipInput.files[0] : null;
    if (!file) {
      setStatus('Please choose a ZIP file to integrate.');
      return;
    }

    setStatus('Analyzing ZIP...');
    log(`Reading ZIP: ${file.name} (${file.size} bytes)`);

    try {
      if (!window.JSZip) {
        setStatus('JSZip failed to load. Check network or ensure ./vendor/jszip.min.js exists.');
        log('JSZip not available on window.');
        return;
      }
      const jszip = new JSZip();
      const zip = await jszip.loadAsync(file);

      const { contentDir, indexPath } = findContentDirFromZip(zip);
      if (!indexPath) {
        setStatus('index.html not found in ZIP.');
        log('Error: index.html not found.');
        return;
      }
      log(`Detected content dir: "${contentDir || '(root)'}", index: "${indexPath}"`);

      const html = await zip.file(indexPath).async('string');

      // Try to load JavaScript file for analysis
      let jsContent = '';
      const jsFiles = Object.keys(zip.files).filter(k => k.endsWith('.js') && !k.includes('node_modules'));
      if (jsFiles.length > 0) {
        try {
          jsContent = await zip.file(jsFiles[0]).async('string');
        } catch (e) {
          log('Could not read JavaScript file for analysis');
        }
      }

      // Analyze and recommend mode
      setStatus('Analyzing interactive content...');
      const analysis = await analyzeInteractive(html, jsContent);
      
      log(`\n📊 ANALYSIS RESULTS:`);
      log(`Recommended Mode: ${analysis.recommendedMode.toUpperCase()} (${analysis.confidence}% confidence)`);
      log(`Detected features:`);
      if (analysis.hasRadioButtons) log('  ✓ Radio buttons');
      if (analysis.hasCheckboxes) log('  ✓ Checkboxes');
      if (analysis.hasTextInputs) log('  ✓ Text inputs');
      if (analysis.hasDragDrop) log('  ✓ Drag & Drop');
      if (analysis.hasGameState) log('  ✓ Game state');
      if (analysis.hasScore) log('  ✓ Scoring system');
      if (analysis.hasCanvas) log('  ✓ Canvas elements');
      if (analysis.hasPlayPause) log('  ✓ Play/Pause controls');
      log(`Reasons: ${analysis.reasons.join(', ')}\n`);

      // Check if user's selected mode matches recommendation
      const userMode = getSelectedMode();
      let finalMode = userMode;

      if (userMode !== analysis.recommendedMode && analysis.confidence >= 60) {
        const shouldChange = confirm(
          `⚠️ MODE RECOMMENDATION\n\n` +
          `Detected: ${analysis.recommendedMode.toUpperCase()} mode (${analysis.confidence}% confidence)\n` +
          `Currently selected: ${userMode.toUpperCase()} mode\n\n` +
          `Reasons:\n${analysis.reasons.map(r => '• ' + r).join('\n')}\n\n` +
          `Would you like to use the recommended ${analysis.recommendedMode.toUpperCase()} mode instead?\n\n` +
          `(Click OK to use recommended mode, Cancel to keep your selection)`
        );
        
        if (shouldChange) {
          finalMode = analysis.recommendedMode;
          // Update the UI to reflect the change
          document.querySelectorAll('input[name="mode"]').forEach(radio => {
            if (radio.value === finalMode) radio.checked = true;
          });
          log(`✓ Switched to recommended mode: ${finalMode.toUpperCase()}`);
        } else {
          log(`User chose to keep ${userMode.toUpperCase()} mode`);
        }
      } else if (userMode === analysis.recommendedMode) {
        log(`✓ Selected mode matches recommendation`);
      }

      const keepAnalytics = shouldKeepAnalytics();
      setStatus('Injecting xAPI integration...');
      log(`Injecting (mode=${finalMode}, keepAnalytics=${keepAnalytics})...`);
      const injectedHtml = injectScriptsIntoHtml(html, { mode: finalMode, keepAnalytics });

      // Write back modified index.html
      zip.file(indexPath, injectedHtml);

      // Load vendors from our public/vendor and add to ZIP under contentDir/lib/
      const vendors = await loadVendors();
      const libBase = joinPath(contentDir, 'lib/');
      zip.file(joinPath(libBase, 'xapiwrapper.min.js'), vendors.wrapper);
      zip.file(joinPath(libBase, 'xAPI.js'), vendors.glue);

      setStatus('Packaging ZIP...');
      const outBlob = await zip.generateAsync({ type: 'blob' });
      const baseName = file.name.replace(/\.zip$/i, '');
      const outName = `integrated_${baseName}.zip`;

      const url = URL.createObjectURL(outBlob);
      downloadLink.href = url;
      downloadLink.download = outName;
      downloadLink.style.display = 'inline-block';
      setStatus('Done. Click "Download ZIP".');
      log(`Generated output: ${outName}`);
    } catch (e) {
      console.error(e);
      setStatus('Integration failed. See log.');
      log(`Error: ${e && e.message ? e.message : String(e)}`);
    }
  });
})();
