// Projectile Motion Interactive - Main JavaScript File
// This file handles all the interactive functionality, physics calculations, and real-time updates

class ProjectileMotion {
    constructor() {
        // Initialize canvas and contexts
        this.canvas = document.getElementById('simulation-canvas');
        this.ctx = this.canvas.getContext('2d');
        
        // Graph canvases
        this.ytGraph = document.getElementById('y-t-graph');
        this.vytGraph = document.getElementById('vy-t-graph');
        this.aytGraph = document.getElementById('ay-t-graph');
        
        this.ytCtx = this.ytGraph.getContext('2d');
        this.vytCtx = this.vytGraph.getContext('2d');
        this.aytCtx = this.aytGraph.getContext('2d');
        
        // Physics parameters
        this.params = {
            velocity: 20,      // m/s
            angle: 45,         // degrees
            gravity: 9.8,      // m/s²
            airResistance: false,
            dragCoeff: 0.01
        };
        
        // Simulation state
        this.state = {
            isRunning: false,
            isPaused: false,
            time: 0,
            projectile: { x: 0, y: 0, vx: 0, vy: 0 },
            trail: [],
            dataLog: [],
            maxHeight: 0,
            range: 0,
            flightTime: 0
        };
        
        // Comparison mode
        this.compareMode = false;
        this.comparisonTrail = [];
        
        // Analytics
        this.actionLog = [];
        this.startTime = Date.now();
        
        // UI elements
        this.elements = this.initializeElements();
        
        // Initialize
        this.setupEventListeners();
        this.setupTooltips();
        this.resizeCanvas();
        this.reset();
        this.logAction('🚀 Simulation initialized', 'System ready');
        
        // Start animation loop
        this.animate();
    }
    
    // Initialize all UI elements
    initializeElements() {
        return {
            velocitySlider: document.getElementById('velocity-slider'),
            angleSlider: document.getElementById('angle-slider'),
            gravitySlider: document.getElementById('gravity-slider'),
            velocityValue: document.getElementById('velocity-value'),
            angleValue: document.getElementById('angle-value'),
            gravityValue: document.getElementById('gravity-value'),
            airResistanceCheck: document.getElementById('air-resistance'),
            showVectorsCheck: document.getElementById('show-vectors'),
            showTrailCheck: document.getElementById('show-trail'),
            compareModeCheck: document.getElementById('compare-mode'),
            launchBtn: document.getElementById('launch-btn'),
            pauseBtn: document.getElementById('pause-btn'),
            resetBtn: document.getElementById('reset-btn'),
            exportBtn: document.getElementById('export-csv'),
            clearLogBtn: document.getElementById('clear-log'),
            toggleAnalyticsBtn: document.getElementById('toggle-analytics'),
            flightTimeDisplay: document.getElementById('flight-time'),
            maxHeightDisplay: document.getElementById('max-height'),
            rangeDisplay: document.getElementById('range'),
            dataTable: document.getElementById('data-tbody'),
            actionLog: document.getElementById('action-log'),
            analyticsContent: document.getElementById('analytics-content'),
            presetBtns: document.querySelectorAll('.preset-btn')
        };
    }
    
    // Setup all event listeners
    setupEventListeners() {
        // Slider events
        this.elements.velocitySlider.addEventListener('input', (e) => {
            this.params.velocity = parseFloat(e.target.value);
            this.elements.velocityValue.textContent = this.params.velocity;
            this.logAction('⚡ Velocity changed', `v₀ = ${this.params.velocity} m/s`);
            this.updateCalculations();
        });
        
        this.elements.angleSlider.addEventListener('input', (e) => {
            this.params.angle = parseFloat(e.target.value);
            this.elements.angleValue.textContent = this.params.angle;
            this.logAction('📐 Angle changed', `θ = ${this.params.angle}°`);
            this.updatePresetButtons();
            this.updateCalculations();
        });
        
        this.elements.gravitySlider.addEventListener('input', (e) => {
            this.params.gravity = parseFloat(e.target.value);
            this.elements.gravityValue.textContent = this.params.gravity;
            this.logAction('🌍 Gravity changed', `g = ${this.params.gravity} m/s²`);
            this.updateCalculations();
        });
        
        // Checkbox events
        this.elements.airResistanceCheck.addEventListener('change', (e) => {
            this.params.airResistance = e.target.checked;
            this.logAction('💨 Air resistance', e.target.checked ? 'enabled' : 'disabled');
            this.updateCalculations();
        });
        
        this.elements.showVectorsCheck.addEventListener('change', (e) => {
            this.logAction('🏹 Vector display', e.target.checked ? 'enabled' : 'disabled');
        });
        
        this.elements.showTrailCheck.addEventListener('change', (e) => {
            this.logAction('✨ Trail display', e.target.checked ? 'enabled' : 'disabled');
        });
        
        this.elements.compareModeCheck.addEventListener('change', (e) => {
            this.compareMode = e.target.checked;
            if (this.compareMode && this.state.trail.length > 0) {
                this.comparisonTrail = [...this.state.trail];
            }
            this.logAction('🔄 Compare mode', e.target.checked ? 'enabled' : 'disabled');
        });
        
        // Button events
        this.elements.launchBtn.addEventListener('click', () => this.launch());
        this.elements.pauseBtn.addEventListener('click', () => this.togglePause());
        this.elements.resetBtn.addEventListener('click', () => this.reset());
        this.elements.exportBtn.addEventListener('click', () => this.exportCSV());
        this.elements.clearLogBtn.addEventListener('click', () => this.clearDataLog());
        this.elements.toggleAnalyticsBtn.addEventListener('click', () => this.toggleAnalytics());
        
        // Preset button events
        this.elements.presetBtns.forEach(btn => {
            btn.addEventListener('click', (e) => {
                const angle = parseInt(e.target.dataset.angle);
                this.setAngle(angle);
            });
        });
        
        // Touch events for mobile
        this.setupTouchEvents();
        
        // Window resize
        window.addEventListener('resize', () => this.resizeCanvas());
    }
    
    // Setup touch events for mobile compatibility
    setupTouchEvents() {
        const sliders = [this.elements.velocitySlider, this.elements.angleSlider, this.elements.gravitySlider];
        
        sliders.forEach(slider => {
            slider.addEventListener('touchstart', (e) => {
                e.preventDefault();
                slider.focus();
            });
            
            slider.addEventListener('touchmove', (e) => {
                e.preventDefault();
                const touch = e.touches[0];
                const rect = slider.getBoundingClientRect();
                const percent = (touch.clientX - rect.left) / rect.width;
                const min = parseFloat(slider.min);
                const max = parseFloat(slider.max);
                const value = min + (max - min) * Math.max(0, Math.min(1, percent));
                
                slider.value = value;
                slider.dispatchEvent(new Event('input'));
            });
        });
    }
    
    // Setup tooltips
    setupTooltips() {
        const tooltip = document.getElementById('tooltip');
        const tooltipElements = document.querySelectorAll('.tooltip');
        
        tooltipElements.forEach(element => {
            element.addEventListener('mouseenter', (e) => {
                const text = e.target.dataset.tooltip;
                if (text) {
                    tooltip.textContent = text;
                    tooltip.classList.add('show');
                    this.updateTooltipPosition(e, tooltip);
                }
            });
            
            element.addEventListener('mousemove', (e) => {
                this.updateTooltipPosition(e, tooltip);
            });
            
            element.addEventListener('mouseleave', () => {
                tooltip.classList.remove('show');
            });
        });
    }
    
    // Update tooltip position
    updateTooltipPosition(e, tooltip) {
        const x = e.clientX + 10;
        const y = e.clientY - 30;
        tooltip.style.left = x + 'px';
        tooltip.style.top = y + 'px';
    }
    
    // Resize canvas to fit container
    resizeCanvas() {
        const container = this.canvas.parentElement;
        const rect = container.getBoundingClientRect();
        
        // Set canvas size
        this.canvas.width = rect.width - 32; // Account for padding
        this.canvas.height = 200;
        
        // Update scale for physics calculations
        this.scale = {
            x: this.canvas.width / 60,  // 60 meters width
            y: this.canvas.height / 30  // 30 meters height
        };
    }
    
    // Launch projectile
    launch() {
        if (this.state.isRunning && !this.state.isPaused) {
            return; // Already running
        }
        
        this.state.isRunning = true;
        this.state.isPaused = false;
        this.state.time = 0;
        
        // Calculate initial velocity components
        const angleRad = this.params.angle * Math.PI / 180;
        this.state.projectile = {
            x: 0,
            y: 0,
            vx: this.params.velocity * Math.cos(angleRad),
            vy: this.params.velocity * Math.sin(angleRad)
        };
        
        // Clear previous data
        this.state.trail = [];
        this.state.dataLog = [];
        this.clearDataTable();
        
        // Update inquiry step
        this.updateInquiryStep('test');
        
        this.logAction('🚀 Projectile launched', `v₀=${this.params.velocity}m/s, θ=${this.params.angle}°`);
        
        // Update button states
        this.elements.launchBtn.textContent = '🚀 Running...';
        this.elements.launchBtn.disabled = true;
    }
    
    // Toggle pause
    togglePause() {
        if (!this.state.isRunning) return;
        
        this.state.isPaused = !this.state.isPaused;
        this.elements.pauseBtn.textContent = this.state.isPaused ? '▶️ Resume' : '⏸️ Pause';
        this.logAction(this.state.isPaused ? '⏸️ Paused' : '▶️ Resumed', `at t=${this.state.time.toFixed(2)}s`);
    }
    
    // Reset simulation
    reset() {
        this.state.isRunning = false;
        this.state.isPaused = false;
        this.state.time = 0;
        this.state.projectile = { x: 0, y: 0, vx: 0, vy: 0 };
        this.state.trail = [];
        this.state.maxHeight = 0;
        this.state.range = 0;
        this.state.flightTime = 0;
        
        // Reset displays
        this.elements.flightTimeDisplay.textContent = '0.0s';
        this.elements.maxHeightDisplay.textContent = '0.0m';
        this.elements.rangeDisplay.textContent = '0.0m';
        
        // Reset buttons
        this.elements.launchBtn.textContent = '🚀 Launch';
        this.elements.launchBtn.disabled = false;
        this.elements.pauseBtn.textContent = '⏸️ Pause';
        
        // Clear data
        this.clearDataTable();
        this.clearGraphs();
        
        // Update inquiry step
        this.updateInquiryStep('predict');
        
        this.logAction('🔄 Reset', 'Simulation reset to initial state');
    }
    
    // Set angle from preset
    setAngle(angle) {
        this.params.angle = angle;
        this.elements.angleSlider.value = angle;
        this.elements.angleValue.textContent = angle;
        this.updatePresetButtons();
        this.logAction('📐 Preset angle', `θ = ${angle}°`);
        this.updateCalculations();
    }
    
    // Update preset button states
    updatePresetButtons() {
        this.elements.presetBtns.forEach(btn => {
            const angle = parseInt(btn.dataset.angle);
            btn.classList.toggle('active', angle === this.params.angle);
        });
    }
    
    // Update theoretical calculations
    updateCalculations() {
        if (this.state.isRunning) return;
        
        const angleRad = this.params.angle * Math.PI / 180;
        const v0 = this.params.velocity;
        const g = this.params.gravity;
        
        // Theoretical calculations (without air resistance)
        const flightTime = (2 * v0 * Math.sin(angleRad)) / g;
        const maxHeight = (v0 * v0 * Math.sin(angleRad) * Math.sin(angleRad)) / (2 * g);
        const range = (v0 * v0 * Math.sin(2 * angleRad)) / g;
        
        this.state.flightTime = flightTime;
        this.state.maxHeight = maxHeight;
        this.state.range = range;
        
        // Update displays
        this.elements.flightTimeDisplay.textContent = flightTime.toFixed(2) + 's';
        this.elements.maxHeightDisplay.textContent = maxHeight.toFixed(2) + 'm';
        this.elements.rangeDisplay.textContent = range.toFixed(2) + 'm';
    }
    
    // Physics update
    updatePhysics(dt) {
        if (!this.state.isRunning || this.state.isPaused) return;
        
        const p = this.state.projectile;
        
        // Update position
        p.x += p.vx * dt;
        p.y += p.vy * dt;
        
        // Update velocity (gravity)
        p.vy -= this.params.gravity * dt;
        
        // Air resistance
        if (this.params.airResistance) {
            const speed = Math.sqrt(p.vx * p.vx + p.vy * p.vy);
            const dragX = -this.params.dragCoeff * speed * p.vx;
            const dragY = -this.params.dragCoeff * speed * p.vy;
            
            p.vx += dragX * dt;
            p.vy += dragY * dt;
        }
        
        // Update time
        this.state.time += dt;
        
        // Add to trail
        if (this.state.time % 0.1 < dt) { // Every 0.1 seconds
            this.state.trail.push({ x: p.x, y: p.y, t: this.state.time });
        }
        
        // Log data
        if (this.state.time % 0.2 < dt) { // Every 0.2 seconds
            this.logData();
        }
        
        // Check for landing
        if (p.y <= 0 && this.state.time > 0.1) {
            this.state.isRunning = false;
            this.state.range = p.x;
            this.state.flightTime = this.state.time;
            
            // Update displays
            this.elements.rangeDisplay.textContent = this.state.range.toFixed(2) + 'm';
            this.elements.flightTimeDisplay.textContent = this.state.flightTime.toFixed(2) + 's';
            
            // Reset buttons
            this.elements.launchBtn.textContent = '🚀 Launch';
            this.elements.launchBtn.disabled = false;
            this.elements.pauseBtn.textContent = '⏸️ Pause';
            
            // Update inquiry step
            this.updateInquiryStep('explain');
            
            this.logAction('🎯 Landing', `Range: ${this.state.range.toFixed(2)}m, Time: ${this.state.flightTime.toFixed(2)}s`);
        }
        
        // Update max height
        if (p.y > this.state.maxHeight) {
            this.state.maxHeight = p.y;
            this.elements.maxHeightDisplay.textContent = this.state.maxHeight.toFixed(2) + 'm';
        }
    }
    
    // Log data point
    logData() {
        const p = this.state.projectile;
        const dataPoint = {
            t: this.state.time,
            x: p.x,
            y: p.y,
            vx: p.vx,
            vy: p.vy
        };
        
        this.state.dataLog.push(dataPoint);
        this.addDataRow(dataPoint);
    }
    
    // Add row to data table
    addDataRow(data) {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${data.t.toFixed(2)}</td>
            <td>${data.x.toFixed(2)}</td>
            <td>${data.y.toFixed(2)}</td>
            <td>${data.vx.toFixed(2)}</td>
            <td>${data.vy.toFixed(2)}</td>
        `;
        this.elements.dataTable.appendChild(row);
        
        // Scroll to bottom
        const container = this.elements.dataTable.parentElement;
        container.scrollTop = container.scrollHeight;
    }
    
    // Clear data table
    clearDataTable() {
        this.elements.dataTable.innerHTML = '';
    }
    
    // Export data as CSV
    exportCSV() {
        if (this.state.dataLog.length === 0) {
            alert('No data to export. Launch the projectile first!');
            return;
        }
        
        let csv = 'Time(s),X(m),Y(m),Vx(m/s),Vy(m/s)\n';
        this.state.dataLog.forEach(data => {
            csv += `${data.t.toFixed(2)},${data.x.toFixed(2)},${data.y.toFixed(2)},${data.vx.toFixed(2)},${data.vy.toFixed(2)}\n`;
        });
        
        const blob = new Blob([csv], { type: 'text/csv' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'projectile_data.csv';
        a.click();
        URL.revokeObjectURL(url);
        
        this.logAction('📥 Data exported', `${this.state.dataLog.length} data points`);
    }
    
    // Clear data log
    clearDataLog() {
        this.state.dataLog = [];
        this.clearDataTable();
        this.clearGraphs();
        this.logAction('🗑️ Data cleared', 'All logged data removed');
    }
    
    // Toggle analytics panel
    toggleAnalytics() {
        const panel = document.querySelector('.analytics-panel');
        const content = this.elements.analyticsContent;
        
        if (content.style.display === 'none') {
            content.style.display = 'block';
            this.elements.toggleAnalyticsBtn.textContent = '👁️ Hide';
            panel.classList.add('show');
        } else {
            content.style.display = 'none';
            this.elements.toggleAnalyticsBtn.textContent = '👁️ Show';
            panel.classList.remove('show');
        }
    }
    
    // Log user action
    logAction(action, details) {
        const timestamp = ((Date.now() - this.startTime) / 1000).toFixed(1);
        const entry = {
            time: timestamp,
            action: action,
            details: details
        };
        
        this.actionLog.push(entry);
        
        // Add to display
        const logEntry = document.createElement('div');
        logEntry.className = 'log-entry';
        logEntry.innerHTML = `<strong>t=${timestamp}s:</strong> ${action} - ${details}`;
        this.elements.actionLog.appendChild(logEntry);
        
        // Scroll to bottom
        this.elements.actionLog.scrollTop = this.elements.actionLog.scrollHeight;
        
        // Limit log size
        if (this.actionLog.length > 50) {
            this.actionLog.shift();
            this.elements.actionLog.removeChild(this.elements.actionLog.firstChild);
        }
    }
    
    // Update inquiry step
    updateInquiryStep(step) {
        const steps = ['predict', 'test', 'explain'];
        steps.forEach(s => {
            const element = document.getElementById(`${s}-step`);
            element.classList.toggle('active', s === step);
        });
    }
    
    // Render simulation
    render() {
        const ctx = this.ctx;
        const canvas = this.canvas;
        
        // Clear canvas
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        // Draw background
        const gradient = ctx.createLinearGradient(0, 0, 0, canvas.height);
        gradient.addColorStop(0, '#87ceeb');
        gradient.addColorStop(1, '#98fb98');
        ctx.fillStyle = gradient;
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        
        // Draw ground
        ctx.fillStyle = '#8b4513';
        ctx.fillRect(0, canvas.height - 20, canvas.width, 20);
        
        // Draw grid
        this.drawGrid(ctx);
        
        // Draw comparison trail if enabled
        if (this.compareMode && this.comparisonTrail.length > 0) {
            this.drawTrail(ctx, this.comparisonTrail, '#ff6b6b', 0.5);
        }
        
        // Draw current trail
        if (this.elements.showTrailCheck.checked && this.state.trail.length > 0) {
            this.drawTrail(ctx, this.state.trail, '#3498db', 1.0);
        }
        
        // Draw projectile
        if (this.state.isRunning || this.state.projectile.x > 0) {
            this.drawProjectile(ctx);
        }
        
        // Draw vectors
        if (this.elements.showVectorsCheck.checked && (this.state.isRunning || this.state.projectile.x > 0)) {
            this.drawVectors(ctx);
        }
        
        // Draw launch point
        this.drawLaunchPoint(ctx);
        
        // Update graphs
        this.updateGraphs();
    }
    
    // Draw grid
    drawGrid(ctx) {
        ctx.strokeStyle = 'rgba(255, 255, 255, 0.3)';
        ctx.lineWidth = 1;
        
        // Vertical lines (every 5 meters)
        for (let x = 0; x <= 60; x += 5) {
            const canvasX = x * this.scale.x;
            ctx.beginPath();
            ctx.moveTo(canvasX, 0);
            ctx.lineTo(canvasX, this.canvas.height - 20);
            ctx.stroke();
        }
        
        // Horizontal lines (every 2.5 meters)
        for (let y = 0; y <= 30; y += 2.5) {
            const canvasY = this.canvas.height - 20 - (y * this.scale.y);
            ctx.beginPath();
            ctx.moveTo(0, canvasY);
            ctx.lineTo(this.canvas.width, canvasY);
            ctx.stroke();
        }
    }
    
    // Draw trail
    drawTrail(ctx, trail, color, alpha) {
        if (trail.length < 2) return;
        
        ctx.strokeStyle = color;
        ctx.globalAlpha = alpha;
        ctx.lineWidth = 2;
        ctx.beginPath();
        
        for (let i = 0; i < trail.length; i++) {
            const point = trail[i];
            const x = point.x * this.scale.x;
            const y = this.canvas.height - 20 - (point.y * this.scale.y);
            
            if (i === 0) {
                ctx.moveTo(x, y);
            } else {
                ctx.lineTo(x, y);
            }
        }
        
        ctx.stroke();
        ctx.globalAlpha = 1.0;
        
        // Draw trail points
        ctx.fillStyle = color;
        trail.forEach(point => {
            const x = point.x * this.scale.x;
            const y = this.canvas.height - 20 - (point.y * this.scale.y);
            ctx.beginPath();
            ctx.arc(x, y, 2, 0, 2 * Math.PI);
            ctx.fill();
        });
    }
    
    // Draw projectile
    drawProjectile(ctx) {
        const p = this.state.projectile;
        const x = p.x * this.scale.x;
        const y = this.canvas.height - 20 - (p.y * this.scale.y);
        
        // Draw projectile
        ctx.fillStyle = '#e74c3c';
        ctx.beginPath();
        ctx.arc(x, y, 6, 0, 2 * Math.PI);
        ctx.fill();
        
        // Draw outline
        ctx.strokeStyle = '#c0392b';
        ctx.lineWidth = 2;
        ctx.stroke();
    }
    
    // Draw velocity vectors
    drawVectors(ctx) {
        const p = this.state.projectile;
        const x = p.x * this.scale.x;
        const y = this.canvas.height - 20 - (p.y * this.scale.y);
        
        const scale = 3; // Vector scale factor
        
        // Velocity vector (total)
        ctx.strokeStyle = '#2ecc71';
        ctx.lineWidth = 3;
        ctx.beginPath();
        ctx.moveTo(x, y);
        ctx.lineTo(x + p.vx * scale, y - p.vy * scale);
        ctx.stroke();
        
        // Arrow head
        this.drawArrowHead(ctx, x, y, x + p.vx * scale, y - p.vy * scale, '#2ecc71');
        
        // Horizontal component
        ctx.strokeStyle = '#3498db';
        ctx.lineWidth = 2;
        ctx.setLineDash([5, 5]);
        ctx.beginPath();
        ctx.moveTo(x, y);
        ctx.lineTo(x + p.vx * scale, y);
        ctx.stroke();
        
        // Vertical component
        ctx.strokeStyle = '#e74c3c';
        ctx.beginPath();
        ctx.moveTo(x, y);
        ctx.lineTo(x, y - p.vy * scale);
        ctx.stroke();
        
        ctx.setLineDash([]);
        
        // Labels
        ctx.fillStyle = '#2c3e50';
        ctx.font = '12px Arial';
        ctx.fillText(`v = ${Math.sqrt(p.vx*p.vx + p.vy*p.vy).toFixed(1)} m/s`, x + 10, y - 10);
        ctx.fillText(`vₓ = ${p.vx.toFixed(1)} m/s`, x + p.vx * scale + 5, y + 5);
        ctx.fillText(`vᵧ = ${p.vy.toFixed(1)} m/s`, x + 5, y - p.vy * scale);
    }
    
    // Draw arrow head
    drawArrowHead(ctx, x1, y1, x2, y2, color) {
        const angle = Math.atan2(y2 - y1, x2 - x1);
        const headLength = 10;
        
        ctx.fillStyle = color;
        ctx.beginPath();
        ctx.moveTo(x2, y2);
        ctx.lineTo(x2 - headLength * Math.cos(angle - Math.PI/6), y2 - headLength * Math.sin(angle - Math.PI/6));
        ctx.lineTo(x2 - headLength * Math.cos(angle + Math.PI/6), y2 - headLength * Math.sin(angle + Math.PI/6));
        ctx.closePath();
        ctx.fill();
    }
    
    // Draw launch point
    drawLaunchPoint(ctx) {
        ctx.fillStyle = '#f39c12';
        ctx.beginPath();
        ctx.arc(0, this.canvas.height - 20, 8, 0, 2 * Math.PI);
        ctx.fill();
        
        ctx.strokeStyle = '#e67e22';
        ctx.lineWidth = 2;
        ctx.stroke();
    }
    
    // Update graphs
    updateGraphs() {
        this.updateYTGraph();
        this.updateVyTGraph();
        this.updateAyTGraph();
    }
    
    // Update Y vs T graph
    updateYTGraph() {
        const ctx = this.ytCtx;
        const canvas = this.ytGraph;
        
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        // Draw axes
        this.drawGraphAxes(ctx, canvas, 'Y (m)', 'Time (s)');
        
        if (this.state.dataLog.length < 2) return;
        
        // Find max values for scaling
        const maxT = Math.max(...this.state.dataLog.map(d => d.t));
        const maxY = Math.max(...this.state.dataLog.map(d => d.y));
        
        // Draw data
        ctx.strokeStyle = '#e74c3c';
        ctx.lineWidth = 2;
        ctx.beginPath();
        
        this.state.dataLog.forEach((data, i) => {
            const x = (data.t / maxT) * (canvas.width - 40) + 30;
            const y = canvas.height - 30 - (data.y / maxY) * (canvas.height - 50);
            
            if (i === 0) {
                ctx.moveTo(x, y);
            } else {
                ctx.lineTo(x, y);
            }
        });
        
        ctx.stroke();
    }
    
    // Update Vy vs T graph
    updateVyTGraph() {
        const ctx = this.vytCtx;
        const canvas = this.vytGraph;
        
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        // Draw axes
        this.drawGraphAxes(ctx, canvas, 'Vᵧ (m/s)', 'Time (s)');
        
        if (this.state.dataLog.length < 2) return;
        
        // Find max values for scaling
        const maxT = Math.max(...this.state.dataLog.map(d => d.t));
        const maxVy = Math.max(...this.state.dataLog.map(d => Math.abs(d.vy)));
        
        // Draw zero line
        ctx.strokeStyle = '#bdc3c7';
        ctx.lineWidth = 1;
        ctx.beginPath();
        const zeroY = canvas.height - 30 - (0 + maxVy) / (2 * maxVy) * (canvas.height - 50);
        ctx.moveTo(30, zeroY);
        ctx.lineTo(canvas.width - 10, zeroY);
        ctx.stroke();
        
        // Draw data
        ctx.strokeStyle = '#3498db';
        ctx.lineWidth = 2;
        ctx.beginPath();
        
        this.state.dataLog.forEach((data, i) => {
            const x = (data.t / maxT) * (canvas.width - 40) + 30;
            const y = canvas.height - 30 - (data.vy + maxVy) / (2 * maxVy) * (canvas.height - 50);
            
            if (i === 0) {
                ctx.moveTo(x, y);
            } else {
                ctx.lineTo(x, y);
            }
        });
        
        ctx.stroke();
    }
    
    // Update Ay vs T graph
    updateAyTGraph() {
        const ctx = this.aytCtx;
        const canvas = this.aytGraph;
        
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        // Draw axes
        this.drawGraphAxes(ctx, canvas, 'aᵧ (m/s²)', 'Time (s)');
        
        // Draw constant acceleration line
        ctx.strokeStyle = '#9b59b6';
        ctx.lineWidth = 2;
        ctx.beginPath();
        
        const accelY = canvas.height - 30 - (this.params.gravity / 20) * (canvas.height - 50);
        ctx.moveTo(30, accelY);
        ctx.lineTo(canvas.width - 10, accelY);
        ctx.stroke();
        
        // Label
        ctx.fillStyle = '#2c3e50';
        ctx.font = '10px Arial';
        ctx.fillText(`-${this.params.gravity} m/s²`, 35, accelY - 5);
    }
    
    // Draw graph axes
    drawGraphAxes(ctx, canvas, yLabel, xLabel) {
        ctx.strokeStyle = '#2c3e50';
        ctx.lineWidth = 1;
        
        // Y axis
        ctx.beginPath();
        ctx.moveTo(30, 10);
        ctx.lineTo(30, canvas.height - 30);
        ctx.stroke();
        
        // X axis
        ctx.beginPath();
        ctx.moveTo(30, canvas.height - 30);
        ctx.lineTo(canvas.width - 10, canvas.height - 30);
        ctx.stroke();
        
        // Labels
        ctx.fillStyle = '#2c3e50';
        ctx.font = '10px Arial';
        ctx.textAlign = 'center';
        ctx.fillText(xLabel, canvas.width / 2, canvas.height - 5);
        
        ctx.save();
        ctx.translate(10, canvas.height / 2);
        ctx.rotate(-Math.PI / 2);
        ctx.fillText(yLabel, 0, 0);
        ctx.restore();
        
        ctx.textAlign = 'left';
    }
    
    // Clear all graphs
    clearGraphs() {
        [this.ytCtx, this.vytCtx, this.aytCtx].forEach(ctx => {
            ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
        });
    }
    
    // Main animation loop
    animate() {
        // Update physics
        this.updatePhysics(0.02); // 20ms timestep
        
        // Render
        this.render();
        
        // Continue animation
        requestAnimationFrame(() => this.animate());
    }
}

// Initialize the simulation when the page loads
document.addEventListener('DOMContentLoaded', () => {
    new ProjectileMotion();
});